/*
 * Decompiled with CFR 0.152.
 */
package edu.sysu.pmglab.kgga.command;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public final class TaskTracker {
    public static final File WORKSPACE_PATH = new File(System.getProperty("user.dir"));
    public static final File DEFAULT_CACHE = new File(System.getProperty("ccf.cache", new File(WORKSPACE_PATH, ".cache").getPath()));
    private static final String HEADER = "TaskID\tInputMD5\tParamterMD5\tOutputPath";
    private static final int EXPECTED_COLUMNS = 4;
    private final Path trackFilePath;
    private final Map<TaskKey, String> completedTasks;

    public TaskTracker(String trackFilePathStr) throws IOException {
        this.trackFilePath = Paths.get(trackFilePathStr, new String[0]);
        this.completedTasks = new HashMap<TaskKey, String>();
        this.loadTrackFile();
    }

    public TaskTracker() throws IOException {
        DEFAULT_CACHE.mkdirs();
        if (!DEFAULT_CACHE.exists()) {
            throw new IllegalStateException(DEFAULT_CACHE + " cannot be written");
        }
        File tempFile = new File(DEFAULT_CACHE, "GlobalTaskTracker.tsv");
        this.trackFilePath = tempFile.toPath();
        this.completedTasks = new HashMap<TaskKey, String>();
        this.loadTrackFile();
    }

    private void loadTrackFile() throws IOException {
        if (!Files.exists(this.trackFilePath, new LinkOption[0])) {
            try (BufferedWriter writer = Files.newBufferedWriter(this.trackFilePath, StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW);){
                writer.write(HEADER);
                writer.newLine();
            }
            return;
        }
        this.completedTasks.clear();
        try (BufferedReader reader = Files.newBufferedReader(this.trackFilePath, StandardCharsets.UTF_8);){
            String line = reader.readLine();
            if (line == null || !line.trim().equals(HEADER)) {
                SetupApplication.GlobalLogger.info("Header mismatch or empty file. Re-initializing track file: " + this.trackFilePath, new Object[0]);
                try (BufferedWriter writer = Files.newBufferedWriter(this.trackFilePath, StandardCharsets.UTF_8, new OpenOption[0]);){
                    writer.write(HEADER);
                    writer.newLine();
                }
                return;
            }
            while ((line = reader.readLine()) != null) {
                String[] columns;
                if (line.trim().isEmpty() || (columns = line.split("\t")).length < 4) continue;
                String taskID = columns[0];
                String inputMD5 = columns[1];
                String parameterMD5 = columns[2];
                String outputPath = columns[3];
                if (outputPath == null || outputPath.trim().isEmpty()) continue;
                TaskKey key = new TaskKey(taskID, inputMD5, parameterMD5);
                this.completedTasks.values().remove(outputPath);
                this.completedTasks.put(key, outputPath);
            }
        }
    }

    public Optional<File> checkTask(TaskResult taskDefinition) throws IOException {
        Objects.requireNonNull(taskDefinition, "TaskResult object cannot be null.");
        return this.checkTask(taskDefinition.getTaskID(), taskDefinition.getInputMD5(), taskDefinition.getParameterMD5());
    }

    public Optional<File> checkTask(String taskID, String inputMD5, String parameterMD5) throws IOException {
        TaskKey key = new TaskKey(taskID, inputMD5, parameterMD5);
        String outputPathStr = this.completedTasks.get(key);
        if (outputPathStr == null || outputPathStr.trim().isEmpty()) {
            return Optional.empty();
        }
        Path outputPath = Paths.get(outputPathStr, new String[0]);
        if (Files.exists(outputPath, new LinkOption[0])) {
            SetupApplication.GlobalLogger.info("Use previous intermediate data in {} for {}.", outputPath.toFile().getCanonicalPath(), taskID.substring(taskID.lastIndexOf(46) + 1));
            return Optional.of(outputPath.toFile());
        }
        return Optional.empty();
    }

    public synchronized void recordTaskCompletion(TaskResult result) throws IOException {
        Objects.requireNonNull(result, "TaskResult object cannot be null.");
        TaskKey newKey = new TaskKey(result.getTaskID(), result.getInputMD5(), result.getParameterMD5());
        String newOutputPath = result.getOutputPath();
        Optional<TaskKey> keyWithSamePath = this.completedTasks.entrySet().stream().filter(entry -> ((String)entry.getValue()).equals(newOutputPath)).map(Map.Entry::getKey).findFirst();
        boolean keyExists = this.completedTasks.containsKey(newKey);
        if (keyExists && newOutputPath.equals(this.completedTasks.get(newKey))) {
            SetupApplication.GlobalLogger.info("Task " + result.getTaskID() + " is already recorded correctly. No update needed.", new Object[0]);
            return;
        }
        Path tempFile = Files.createTempFile(this.trackFilePath.getParent(), "track-", ".tmp", new FileAttribute[0]);
        try (BufferedReader reader = Files.newBufferedReader(this.trackFilePath, StandardCharsets.UTF_8);
             BufferedWriter writer = Files.newBufferedWriter(tempFile, StandardCharsets.UTF_8, new OpenOption[0]);){
            String line;
            writer.write(HEADER);
            writer.newLine();
            reader.readLine();
            while ((line = reader.readLine()) != null) {
                String[] columns;
                if (line.trim().isEmpty() || (columns = line.split("\t")).length < 4) continue;
                TaskKey lineKey = new TaskKey(columns[0], columns[1], columns[2]);
                String lineOutputPath = columns[3];
                if (lineKey.equals(newKey) || lineOutputPath.equals(newOutputPath)) continue;
                writer.write(line);
                writer.newLine();
            }
            String newRecordString = String.join((CharSequence)"\t", result.getTaskID(), result.getInputMD5(), result.getParameterMD5(), result.getOutputPath());
            writer.write(newRecordString);
            writer.newLine();
        }
        Files.move(tempFile, this.trackFilePath, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        if (keyWithSamePath.isPresent()) {
            this.completedTasks.remove(keyWithSamePath.get());
        }
        this.completedTasks.put(newKey, newOutputPath);
    }

    public static void main(String[] args) {
        try {
            String trackFile = "analysis_track.tsv";
            Files.deleteIfExists(Paths.get(trackFile, new String[0]));
            TaskTracker tracker = new TaskTracker(trackFile);
            System.out.println("1. Tracker \u521d\u59cb\u5316\u5b8c\u6210\u3002");
            System.out.println("\u521d\u59cb\u6587\u4ef6\u5185\u5bb9:");
            Files.readAllLines(Paths.get(trackFile, new String[0])).forEach(System.out::println);
            System.out.println("----------------------------------------");
            TaskResult taskA = new TaskResult("TaskA", "in_md5_A", "param_md5_A", "output/result_A.txt");
            tracker.recordTaskCompletion(taskA);
            System.out.println("2. \u8bb0\u5f55 TaskA \u5b8c\u6210\u3002");
            System.out.println("\u5f53\u524d\u6587\u4ef6\u5185\u5bb9:");
            Files.readAllLines(Paths.get(trackFile, new String[0])).forEach(System.out::println);
            System.out.println("----------------------------------------");
            TaskResult taskB = new TaskResult("TaskB", "in_md5_B", "param_md5_B", "output/result_B.txt");
            tracker.recordTaskCompletion(taskB);
            System.out.println("3. \u8bb0\u5f55 TaskB \u5b8c\u6210\u3002");
            System.out.println("\u5f53\u524d\u6587\u4ef6\u5185\u5bb9:");
            Files.readAllLines(Paths.get(trackFile, new String[0])).forEach(System.out::println);
            System.out.println("----------------------------------------");
            System.out.println("4. \u51c6\u5907\u8bb0\u5f55 TaskC\uff0c\u5b83\u7684\u8f93\u51fa\u8def\u5f84\u4e0e TaskA \u76f8\u540c...");
            TaskResult taskC_replaces_A = new TaskResult("TaskC", "in_md5_C", "param_md5_C", "output/result_A.txt");
            tracker.recordTaskCompletion(taskC_replaces_A);
            System.out.println("\u8bb0\u5f55 TaskC \u5b8c\u6210\u3002TaskA \u7684\u8bb0\u5f55\u5e94\u8be5\u88ab TaskC \u66ff\u6362\u3002");
            System.out.println("\u5f53\u524d\u6587\u4ef6\u5185\u5bb9:");
            Files.readAllLines(Paths.get(trackFile, new String[0])).forEach(System.out::println);
            System.out.println("----------------------------------------");
            System.out.println("5. \u51c6\u5907\u8bb0\u5f55 TaskD\uff0c\u5b83\u7684\u4efb\u52a1Key\u4e0e TaskB \u76f8\u540c\uff0c\u4f46 outputPath \u4e0d\u540c...");
            TaskResult taskD_replaces_B = new TaskResult("TaskB", "in_md5_B", "param_md5_B", "output/result_D_final.txt");
            tracker.recordTaskCompletion(taskD_replaces_B);
            System.out.println("\u8bb0\u5f55 TaskD \u5b8c\u6210\u3002TaskB \u7684\u8bb0\u5f55\u5e94\u8be5\u88ab TaskD \u66f4\u65b0\u3002");
            System.out.println("\u6700\u7ec8\u6587\u4ef6\u5185\u5bb9:");
            Files.readAllLines(Paths.get(trackFile, new String[0])).forEach(System.out::println);
            long lineCount = Files.lines(Paths.get(trackFile, new String[0])).count();
            if (lineCount == 3L) {
                System.out.println("\n\u9a8c\u8bc1\u6210\u529f\uff1a\u6587\u4ef6\u5305\u542b\u6b63\u786e\u6570\u91cf\u7684\u8bb0\u5f55\u3002");
            } else {
                System.err.println("\n\u9a8c\u8bc1\u5931\u8d25\uff1a\u6587\u4ef6\u8bb0\u5f55\u6570\u91cf\u4e0d\u6b63\u786e\uff01");
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    static class SetupApplication {
        SetupApplication() {
        }

        static final class GlobalLogger {
            GlobalLogger() {
            }

            static void info(String format, Object ... args) {
                String message = String.format(format.replace("{}", "%s"), args);
            }
        }
    }

    public static final class TaskResult {
        private final String taskID;
        private final String inputMD5;
        private final String parameterMD5;
        private String outputPath;

        public TaskResult(String taskID, String inputMD5, String parameterMD5, String outputPath) {
            this.taskID = taskID;
            this.inputMD5 = inputMD5;
            this.parameterMD5 = parameterMD5;
            this.outputPath = outputPath;
        }

        public TaskResult(String taskID, String inputMD5, String parameterMD5) {
            this.taskID = taskID;
            this.inputMD5 = inputMD5;
            this.parameterMD5 = parameterMD5;
        }

        public String getTaskID() {
            return this.taskID;
        }

        public String getInputMD5() {
            return this.inputMD5;
        }

        public String getParameterMD5() {
            return this.parameterMD5;
        }

        public String getOutputPath() {
            return this.outputPath;
        }

        public void setOutputPath(File outputFile) throws IOException {
            this.outputPath = outputFile.getCanonicalPath();
        }
    }

    public static final class TaskKey {
        private final String taskID;
        private final String inputMD5;
        private final String parameterMD5;

        public TaskKey(String taskID, String inputMD5, String parameterMD5) {
            this.taskID = Objects.requireNonNull(taskID);
            this.inputMD5 = Objects.requireNonNull(inputMD5);
            this.parameterMD5 = Objects.requireNonNull(parameterMD5);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TaskKey taskKey = (TaskKey)o;
            return this.taskID.equals(taskKey.taskID) && this.inputMD5.equals(taskKey.inputMD5) && this.parameterMD5.equals(taskKey.parameterMD5);
        }

        public int hashCode() {
            return Objects.hash(this.taskID, this.inputMD5, this.parameterMD5);
        }
    }
}

